using System.Collections;
using UnityEngine;
using MathNet.Numerics.Distributions;
using UnityEngine.SceneManagement;
using UnityEngine.UI;

/*
 * Manager for the Visualised Liquid Democracy Simulation Environment
 */
public class LiquidDemocracy : MonoBehaviour
{
    int numRuns;
    int run = 0;
    bool oneShot;

    int numVoters;
    public VoterHandler voterPrefab;
    VoterHandler[] voters;

    public Camera camera;
    Vector3 cameraStart;

    public TextMesh optionA;
    public TextMesh optionB;
    Color colourA;
    Color colourB;
    int correctOption;

    public Text roundText;
    public Text progressText;
    bool progress = false;
    
    public Image agentStatsScreen;
    bool displaying = false;
    VoterHandler displayedAgent;
    public Text voterIDText;
    public Text typeText;
    public Text competenceText;
    public Text perceivedText;
    public Text powerText;
    public Text correctText;
    public Text incorrectText;
    public Text delegateText;
    public Text guruText;
    public Text delegatorsText;
    public Text directText;

    // Start is called before the first frame update, Simualtion environment put in start state
    void Start()
    {
        numVoters = StatsManager.instance.numVoters;
        numRuns = StatsManager.instance.numRounds;

        if(StatsManager.instance.roundType == 0) { oneShot = true; }
        else { oneShot = false; }

        if (StatsManager.instance.largeCommunity)
        {
            camera.transform.position = new Vector3(0, 20.2f, -34);
        }

        cameraStart = camera.transform.position;
        colourA = optionA.color;
        colourB = optionB.color;

        roundText.text = "Voting Round: " + (run + 1) + "/" + numRuns;
        progressText.enabled = false;

        voters = new VoterHandler[numVoters];
        StartCoroutine(Spawn());
    }


    // One Shot: Simulation moved on to next round or ends
    void NextRun()
    {
        run++;
        HideAgentStats();
        roundText.text = "Voting Round: " + (run + 1) + "/" + numRuns;

        // If set number of rounds not completed move on to next with new voters
        if (run < numRuns)
        {
            camera.transform.position = cameraStart;
            foreach (VoterHandler voter in voters)
            {
                voter.Remove();
            }

            optionA.text = "A";
            optionA.color = colourA;
            optionB.text = "B";
            optionB.color = colourB;

            voters = new VoterHandler[numVoters];
            StartCoroutine(Spawn());
        }
        // Else end simualtion and go to statistics screen
        else
        {
            SceneManager.LoadScene("StatsScene");
        }
    }


    // Learning Agents: Simulation moves on to next vote or ends
    IEnumerator NextVote()
    {
        run++;
        roundText.text = "Voting Round: " + (run + 1) + "/" + numRuns;

        // If desired number of votes not completed move voters back to start decision and start next voting round
        if (run < numRuns)
        {
            camera.transform.position = cameraStart;

            foreach(VoterHandler voter in voters)
            {
                voter.ReturnToStart();
            }

            bool doneReturning = false;
            while (!doneReturning)
            {
                doneReturning = true;
                for (int i = 0; i < numVoters; i++)
                {
                    if (voters[i].moving == true)
                    {
                        doneReturning = false;
                    }
                }
                yield return new WaitForSeconds(1.0f / StatsManager.instance.speed);
            }

            optionA.text = "A";
            optionA.color = colourA;
            optionB.text = "B";
            optionB.color = colourB;

            yield return new WaitForSeconds(1.0f / StatsManager.instance.speed);

            if (StatsManager.instance.analysis)
            {
                progressText.enabled = true;
                progress = false;
                while (!progress)
                {
                    yield return new WaitForSeconds(0.1f);
                }
            }
            progressText.enabled = false;

            VoterHandler[] votersByComp = new VoterHandler[numVoters];
            System.Array.Copy(voters, votersByComp, numVoters);
            System.Array.Sort(votersByComp, delegate (VoterHandler x, VoterHandler y) { return x.percievedCompetence.CompareTo(y.percievedCompetence); });
            System.Array.Reverse(votersByComp);

            correctOption = Random.Range(0, 2);
            foreach (VoterHandler voter in voters)
            {
                if (voter.IsGuru())
                {
                    voter.ReviewVote(votersByComp);
                    yield return new WaitForSeconds(0.2f / StatsManager.instance.speed);
                }

                voter.NewVote(correctOption);
            }

            yield return new WaitForSeconds(1.0f / StatsManager.instance.speed);

            StartCoroutine(Votes());
        }
        // Else end simualtion and show statistics screen
        else
        {
            SceneManager.LoadScene("StatsScene");
        }
    }

    // Spawn in the desired number of voter agents to the environment
    IEnumerator Spawn()
    {
        // Program waits for user to press SPACE in anaylsis mode
        if (StatsManager.instance.analysis)
        {
            progressText.enabled = true;
            progress = false;
            while (!progress)
            {
                yield return new WaitForSeconds(0.1f);
            }
        }
        progressText.enabled = false;

        int voterID = 0;
        var gammaDistribution = new Gamma(1.0, 1.0 / StatsManager.instance.compGain);

        int rowSize;
        if (numVoters < 30) { rowSize = 5; }
        else if (numVoters < 60) { rowSize = 10; }
        else { rowSize = 20; }

        int numRows = (int) System.Math.Ceiling((float)numVoters / rowSize);
        float xStartPos = 3 * -((rowSize / 2.0f) - 0.5f);
        float zStartPos = 3 * ((numRows / 2.0f) - 0.5f);

        // Voters spawned in rows
        for (int i = 0; i < numRows; i++)
        {
            int rowVoters = rowSize;
            if (i == numRows - 1 && numVoters % rowSize != 0)
            {
                rowVoters = numVoters % rowSize;
            }
            for (int j = 0; j < rowVoters; j++)
            {
                yield return new WaitForSeconds(0.1f / StatsManager.instance.speed);
                voters[voterID] = Instantiate(voterPrefab, new Vector3(xStartPos + (j * 3.0f), 2.9f, zStartPos - (i * 3.0f)), Quaternion.identity);

                // Voter setup with random competence generated by gamma distribution
                double gammaComp = gammaDistribution.Sample();
                voters[voterID].SetupVoter(voterID, gammaComp, StatsManager.instance.speed, StatsManager.instance.prefAttach, StatsManager.instance.compImp, StatsManager.instance.alphaValue, this);
                voterID++;
            }
        }
        
        yield return new WaitForSeconds(3.0f / StatsManager.instance.speed);
        StartCoroutine(Delegations());
    }

    // Voters make their first guru and delegator decision, and make delegations if they choose to, delegation lines appear
    IEnumerator Delegations()
    {
        if (StatsManager.instance.analysis)
        {
            progressText.enabled = true;
            progress = false;
            while (!progress)
            {
                yield return new WaitForSeconds(0.1f);
            }
        }
        progressText.enabled = false;

        foreach (VoterHandler voter in voters)
        {
            voter.SetupNetwork(StatsManager.instance.networkSize, voters);
        }

        VoterHandler[] votersByComp = new VoterHandler[numVoters];
        System.Array.Copy(voters, votersByComp, numVoters);

        System.Array.Sort(votersByComp, delegate (VoterHandler x, VoterHandler y) { return x.percievedCompetence.CompareTo(y.percievedCompetence); });
        System.Array.Reverse(votersByComp);

        correctOption = Random.Range(0, 2);
        for (int i = 0; i < numVoters; i++)
        {
            yield return new WaitForSeconds(0.1f / StatsManager.instance.speed);
            voters[i].NewVote(correctOption);
            voters[i].GuruOrDelegator(votersByComp);
        }

        yield return new WaitForSeconds(3.0f / StatsManager.instance.speed);
        StartCoroutine(Votes());
    }

    // Gurus make their votes, move to chosen voting area with their delegators
    IEnumerator Votes()
    {
        if (StatsManager.instance.analysis)
        {
            progressText.enabled = true;
            progress = false;
            while (!progress)
            {
                yield return new WaitForSeconds(0.1f);
            }
        }
        progressText.enabled = false;

        int numGurus = 0;
        int highestPower = 0;
        double avgCompetence = 0;

        // Gurus vote with their power and totals for each option are collected
        int right = 0;
        int wrong = 0;
        for (int i = 0; i < numVoters; i++)
        {
            if (voters[i].IsGuru())
            {
                numGurus++;

                int vote;
                int power;

                yield return new WaitForSeconds(0.5f / StatsManager.instance.speed);
                voters[i].CastVote(out vote, out power);
                if (vote == 0) { wrong += power; }
                else { right += power; }

                if (power > highestPower) { highestPower = power; }
                avgCompetence += voters[i].GetTrueCompetence() * power;
            }
        }

        avgCompetence = avgCompetence / numVoters;

        yield return new WaitForSeconds(3.0f / StatsManager.instance.speed);

        // Simulation waits for all voters to move to their chsoen option before displaying results
        bool doneVoting = false;
        while (!doneVoting)
        {
            doneVoting = true;
            for (int i = 0; i < numVoters; i++)
            {
                if (voters[i].moving == true)
                {
                    doneVoting = false;
                }
            }
            yield return new WaitForSeconds(1.0f / StatsManager.instance.speed);
        }

        if (StatsManager.instance.analysis)
        {
            progressText.enabled = true;
            progress = false;
            while (!progress)
            {
                yield return new WaitForSeconds(0.1f);
            }
        }
        progressText.enabled = false;

        camera.transform.position += new Vector3(0, 0, 18);
        yield return new WaitForSeconds(2.0f / StatsManager.instance.speed);


        if (correctOption == 0)
        {
            optionA.text = right.ToString();
            optionB.text = wrong.ToString();
        }
        else
        {
            optionA.text = wrong.ToString();
            optionB.text = right.ToString();
        }

        // Voters animated, celebrate if their chosen option wins, else disappointed
        if (StatsManager.instance.speed <= 2)
        {
            for (int i = 0; i < numVoters; i++)
            {
                if (right > wrong)
                {
                    if (voters[i].vote == 1) { voters[i].Win(); }
                    else { voters[i].Lose(); }
                }
                else if (wrong > right)
                {
                    if (voters[i].vote == 1) { voters[i].Lose(); }
                    else { voters[i].Win(); }
                }
            }
        }

        yield return new WaitForSeconds(3.0f / StatsManager.instance.speed);

        if (correctOption == 0)
        {
            optionA.text = "Right";
            optionA.color = Color.green;
            optionB.text = "Wrong";
            optionB.color = Color.red;
        }
        else
        {
            optionA.text = "Wrong";
            optionA.color = Color.red;
            optionB.text = "Right";
            optionB.color = Color.green;
        }

        // In case of draw, coin flip to decide winning decision
        if (right == wrong)
        {
            int coinFlip = Random.Range(0, 2);
            if (coinFlip == 0)
            {
                wrong++;
            }
            else
            {
                right++;
            }

        }

        // Learning Agents: Perceived Competence of Gurus is updated
        if (StatsManager.instance.roundType == 1)
        {
            foreach (VoterHandler voter in voters)
            {
                voter.UpdatePercievedComp();
            }
        }

        // Voters animated, celebrate if right option wins, else disappointed
        if (right > wrong)
        {
            StatsManager.instance.correct++;
            foreach(VoterHandler voter in voters)
            {
                if (StatsManager.instance.speed <= 2)
                {
                    voter.Win();
                }
            }
        }
        else
        {
            StatsManager.instance.incorrect++;
            foreach (VoterHandler voter in voters)
            {
                if (StatsManager.instance.speed <= 2)
                {
                    voter.Lose();
                }
            }
        }

        yield return new WaitForSeconds(3.0f / StatsManager.instance.speed);

        if (StatsManager.instance.analysis)
        {
            progressText.enabled = true;
            progress = false;
            while (!progress)
            {
                yield return new WaitForSeconds(0.1f);
            }
        }
        progressText.enabled = false;

        StatsManager.instance.numGurusTotal += numGurus;
        StatsManager.instance.avgCompetenceTotal += avgCompetence;
        StatsManager.instance.highestPowerTotal += highestPower;

        if (oneShot)
        {
            NextRun();
        }
        else
        {
            StartCoroutine(NextVote());
        }
    }

    // Agent information UI appears and shows statistics of clicked agent
    public void DisplayAgentStats(VoterHandler agent)
    {
        agentStatsScreen.GetComponent<RectTransform>().anchoredPosition = new Vector2(330, 0);

        displayedAgent = agent;
        displaying = true;
    }

    // Agent information UI closed
    public void HideAgentStats()
    {
        agentStatsScreen.GetComponent<RectTransform>().anchoredPosition = new Vector2(3300, 0);
        displaying = false;
    }

    // Update is called once per frame
    void Update()
    {
        // user presses space key in analysis mode to move simulation to next stage
        if (Input.GetKey("space"))
        {
            progress = true;
        }

        // UI information updated
        if (displaying)
        {
            voterIDText.text = "VoterID: " + displayedAgent.voterID;

            if (displayedAgent.IsGuru()) { typeText.text = "Type: Guru"; }
            else { typeText.text = "Type: Delegator"; }

            competenceText.text = "Competence: " + Mathf.Round((float)displayedAgent.GetTrueCompetence() * 100) / 100.0;
            perceivedText.text = "Perceived Comp: " + Mathf.Round((float)displayedAgent.percievedCompetence * 100) / 100.0;
            powerText.text = "Power: " + displayedAgent.GetPower();
            correctText.text = "Correct Votes: " + displayedAgent.correctVotes;
            incorrectText.text = "Incorrect Votes: " + displayedAgent.incorrectVotes;
            delegateText.text = "Delegate ID: " + displayedAgent.GetDelegateID();
            guruText.text = "Guru ID: " + displayedAgent.GetGuruID();
            delegatorsText.text = displayedAgent.ListDelegators();
            directText.text = displayedAgent.ListDirectDelegators();
        }
    }
}